//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-10-20, Loftur Jonasson, TF3LJ / VE2LJX
//**
//**
//** History.........: Check the ML.c file
//**
//*********************************************************************************

#include "ML.h"
#include "lcd.h"
#include <stdio.h>


// First Level Menu Items
// Note - number of Frequencies needs to be same as NUM_PRESETS as defined in ML.h
const uint8_t level0_menu_size = 32;
const char *level0_menu_items[] =
				{  "Frequency 1",
				   "Frequency 2",
				   "Frequency 3",
				   "Frequency 4",
				   "Frequency 5",
				   "Frequency 6",
				   "Frequency 7",
				   "Frequency 8",
				   "Frequency 9",
				   "Frequency 10",
   				   "Frequency 11",
				   "Frequency 12",
				   "Frequency 13",
				   "Frequency 14",
				   "Frequency 15",
				   "Frequency 16",
				   "Frequency 17",
				   "Frequency 18",
				   "Frequency 19",
				   "Frequency 20",
   				   "Frequency 21",
   				   "Frequency 22",
   				   "Frequency 23",
   				   "Frequency 24",
   				   "Frequency 25",
   				   "Frequency 26",
   				   "Frequency 27",
   				   "Frequency 28",
   				   "Frequency 29",
   				   "Frequency 30",
				   "Factory Reset",
				   "Exit" };

// Flag for Debug Screen
//#define DEBUG_MENU		1

// Flag for Reference 1 Menu Selection
//#define POW_MENU		2

/*// Flag for Set Gain values
#define GAIN_SEL_MENU		6
const uint8_t gain_sel_size = 6;
const char *gain_sel_items[] =
				{  "1-No Gain",
				   "2-GainPreset 1",
				   "3-GainPreset 2",
				   "4-GainPreset 3",
				   "9-Go Back",
				   "0-Exit"};

// Flag for Set Gain values
#define GAIN_MENU		7
const uint8_t gain_menu_size = 5;
const char *gain_menu_items[] =
				{  "1-GainPreset 1",
				   "2-GainPreset 2",
				   "3-GainPreset 3",
				   "9-Go Back",
				   "0-Exit"};

// Flags for Gain Submenu functions
#define GAIN_SET1_MENU	701
#define GAIN_SET2_MENU	702
#define GAIN_SET3_MENU	703
*/


// Flag for Factory Reset
#define FACTORY_MENU	10
// Factory Reset menu Items
const uint8_t factory_menu_size = 3;
const char *factory_menu_items[] =
				{  "1-Yes - Reset",
				   "2-No  - Go back",
				   "3-No  - Exit"};


uint16_t		menu_level = 0;						// Keep track of which menu we are in
uint8_t			menu_data = 0;						// Pass data to lower menu
char 			lcd_buf[20];


//----------------------------------------------------------------------
// A super inelegant quick and dirty Sort() function for
// Frequency and Position Presets from lowest and up,
// but with zero values in highest positions
//----------------------------------------------------------------------
void preset_sort(void)
{
	uint8_t x, sort;
	int32_t temp;

	// Very slow and inelegant :)
	for (x = 0; x < NUM_PRESETS-1; x++)
	{
		for (sort = 0; sort < NUM_PRESETS-1; sort++)
		{
			// Empty elements go up
			if ((R.presetFrq[sort] == 0) && (R.presetFrq[sort+1] > 0))
			{
				R.presetFrq[sort] = R.presetFrq[sort+1];
				R.presetFrq[sort+1] = 0;
	
				R.presetPos[sort] = R.presetPos[sort+1];
				R.presetPos[sort+1] = 0;
			}
			// Empty elements are not shifted down
			else if (R.presetFrq[sort+1] == 0);
			// Shift elements, one by one
			else if (R.presetFrq[sort] > R.presetFrq[sort+1])
			{
				temp = R.presetFrq[sort];
				R.presetFrq[sort] = R.presetFrq[sort+1];
				R.presetFrq[sort+1] = temp;
	
				temp = R.presetPos[sort];
				R.presetPos[sort] = R.presetPos[sort+1];
				R.presetPos[sort+1] = temp;
			}
		}
	}
	// Set all uncalibrated Preset Positions at same value as first Preset
	for (x = 1; x < NUM_PRESETS; x++)
	{
		if (R.presetFrq[x] == 0)
			R.presetPos[x] = R.presetPos[0];
	}		
}


//----------------------------------------------------------------------
// Display a Menu of choices, one line at a time
//
// **menu refers to a pointer array containing the Menu to be printed
//
// menu_size indicates how many pointers (menu items) there are in the array
//
// current_choice indicates which item is currently up for selection if
// pushbutton is pushed
//
// begin row & begin_col are the coordinates for the upper lefthand corner
// of the three or four lines to be printed
//
//----------------------------------------------------------------------
void lcd_scroll_Menu(char **menu, uint8_t menu_size,
		uint8_t current_choice, uint8_t begin_row, uint8_t begin_col)
{
	uint8_t a;

	// Clear LCD from begin_col to end of line.
	lcdGotoXY(begin_col, begin_row);
	for (a = begin_col; a < 16; a++)
		lcdDataWrite(' ');

	// Using Menu list pointed to by **menu, preformat for print:
	// First line contains previous choice, secon line contains
	// current choice preceded with a '->', and third line contains
	// next choice
	lcdGotoXY(begin_col, begin_row);
	sprintf(lcd_buf,"->%s", *(menu + current_choice));
	lcdPrintData(lcd_buf, strlen(lcd_buf));
}


//----------------------------------------------------------------------
// Menu functions begin:
//----------------------------------------------------------------------

/*
//--------------------------------------------------------------------
// Debug Screen, exit on push
//--------------------------------------------------------------------
void debug_menu(void)
{
	lcd_display_mixed();		// Display diverse debug stuff

	// Exit on Button Push
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;	// Clear pushbutton status

		lcdClear();
		lcdGotoXY(1,1);				
		rprintf("Nothing Changed");
		_delay_ms(500);
		Menu_Mode |= CONFIG;	// We're NOT done, just backing off
		menu_level = 0;			// We are done with this menu level
	}
}
*/




/*
//--------------------------------------------------------------------
// Rotary Encoder Resolution
//--------------------------------------------------------------------
void encoder_menu(void)
{

	uint8_t	current_selection;			// Keep track of current Encoder Resolution

	static uint8_t LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Get Current value
	current_selection = R.encoderRes;

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput > 0)
		{
			current_selection=current_selection<<1;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput < 0)
		{
			current_selection=current_selection>>1;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		if(current_selection > 128) current_selection = 128;
		if(current_selection < 1) current_selection = 1;

		// Store Current value in running storage
		R.encoderRes = current_selection;

		lcdClear();
		lcdGotoXY(0,0);	
		rprintf("Encoder ResDivide:");

		lcdGotoXY(0,1);
		rprintf("Rotate to Adjust");
		lcdGotoXY(0,2);
		rprintf("Push to Save");
		// Format and print current value
		lcdGotoXY(0,3);
		rprintf("->");

		int16_t val = current_selection;
		rprintf("%3u",val);
	}

	// Enact selection by saving in EEPROM
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		// Save modified value
		eeprom_write_block(&R.encoderRes, &E.encoderRes, sizeof (R.encoderRes));

		Status &=  ~SHORT_PUSH;			// Clear pushbutton status
		lcdClear();
		lcdGotoXY(1,1);				
		rprintf("Value Stored");
		_delay_ms(500);
		Menu_Mode |= CONFIG;			// We're NOT done, just backing off
		menu_level = 0;					// We are done with this menu level
		LCD_upd = FALSE;				// Make ready for next time
	}
}
*/




//--------------------------------------------------------------------
// Factory Reset with all default values
//--------------------------------------------------------------------
void factory_menu(void)
{
	static int8_t	current_selection;
	static uint8_t	LCD_upd = FALSE;		// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput/8 > 0)
		{
			current_selection++;
		}
		else if (encOutput/8 < 0)
		{
			current_selection--;
		}
  	  	// Reset data from Encoder
		Status &=  ~ENC_CHANGE;
		encOutput = 0;

		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	// If LCD update is needed
	if (LCD_upd == FALSE)
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = factory_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcdClear();
		lcdGotoXY(0,0);
		lcdPrintData("All to default?",15);

		// Print the Rotary Encoder scroll Menu
		lcd_scroll_Menu((char**)factory_menu_items, menu_size, current_selection, 1, 0);
	}

	// Enact selection
	if (Status & SHORT_PUSH)
	{
		Status &=  ~SHORT_PUSH;			// Clear pushbutton status

		switch (current_selection)
		{
			case 0: // Factory Reset
				// Force an EEPROM update:
				R.EEPROM_init_check = 0;		// Initialize eeprom to "factory defaults by indicating a mismatch
				eeprom_write_block(&R.EEPROM_init_check, &E.EEPROM_init_check, sizeof (R.EEPROM_init_check));
				lcdClear();
				lcdGotoXY(0,0);				
				lcdPrintData("Factory Reset",13);
				lcdGotoXY(0,1);
				lcdPrintData("All default",11);
				while (1);			// Bye bye, Death by Watchdog
			case 1:
				lcdClear();
				lcdGotoXY(1,1);				
				lcdPrintData("Nothing Changed",15);
				_delay_ms(500);
				Menu_Mode |= CONFIG;// We're NOT done, just backing off
				menu_level = 0;		// We are done with this menu level
				LCD_upd = FALSE;	// Make ready for next time
				break;
			default:
				lcdClear();
				lcdGotoXY(1,1);				
				lcdPrintData("Nothing Changed",15);
				_delay_ms(500);
				Menu_Mode &=  ~CONFIG;	// We're done
				menu_level = 0;			// We are done with this menu level
				LCD_upd = FALSE;		// Make ready for next time
				break;
		}
	}
}


//
//--------------------------------------------------------------------
// Manage the first level of Menus
//--------------------------------------------------------------------
//
void menu_level0(void)
{
	static int8_t	current_selection;	// Keep track of current menu selection
	static uint8_t	LCD_upd = FALSE;	// Keep track of LCD update requirements

	// Selection modified by encoder.  We remember last selection, even if exit and re-entry
	if (Status & ENC_CHANGE)
	{
		if (encOutput/8 > 0)
		{
			current_selection++;
	    	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		else if (encOutput/8 < 0)
		{
			current_selection--;
	  	  	// Reset data from Encoder
			Status &=  ~ENC_CHANGE;
			encOutput = 0;
		}
		// Indicate that an LCD update is needed
		LCD_upd = FALSE;
	}

	if (LCD_upd == FALSE)				// Need to update LCD
	{
		LCD_upd = TRUE;					// We have serviced LCD

		// Keep Encoder Selection Within Bounds of the Menu Size
		uint8_t menu_size = level0_menu_size;
		while(current_selection >= menu_size)
			current_selection -= menu_size;
		while(current_selection < 0)
			current_selection += menu_size;

		lcdClear();
		// Print the Menu
		lcd_scroll_Menu((char**)level0_menu_items, menu_size, current_selection,1, 0);
	}
	if (current_selection < NUM_PRESETS)
	{
		lcdGotoXY(0,0);
		lcdPrintData("Calibrate:",10);

		lcdGotoXY(0,2);
		display_frq(R.presetFrq[current_selection]);
		lcdGotoXY(14,2);
		display_stepper_pos(R.presetPos[current_selection]);

		lcdGotoXY(0,3);
		display_frq(R.curFrq);
		lcdGotoXY(14,3);
		display_stepper_pos(stepper_track);
	}
	else
	{
		lcdGotoXY(0,0);
		lcdPrintData("Rotate and select:",18);
	}

	if (Status & SHORT_PUSH)
	{
		Status &= ~SHORT_PUSH;			// Clear pushbutton status

		if (current_selection < NUM_PRESETS)
		{
			// TEMPORARY CODE XXXXXXXXXXXXXXXXX
			// Store Current value in running storage
			R.presetFrq[current_selection] = R.curFrq;
			// Set following positions
			R.curPos = R.curPos + delta_Pos;
			stepper_track = R.curPos;
			delta_Pos = 0;
			R.presetPos[current_selection] = R.curPos;

			// Sort all presets in an ascending order,
			// but with empty poitions on top
			preset_sort();
			
			// Store the whole block of frequencies and positions in eeprom
			//eeprom_write_block(&R.presetFrq[current_selection], 
			//	&E.presetFrq[current_selection], 
			//	sizeof (R.presetFrq[current_selection]));
			//eeprom_write_block(&R.presetPos[current_selection], 
			//	&E.presetPos[current_selection], 
			//	sizeof (R.presetPos[current_selection]));
			uint8_t x;
			for (x = 0; x < NUM_PRESETS; x++)
			{
				eeprom_write_block(&R.presetFrq[x],
					&E.presetFrq[x],
					sizeof (R.presetFrq[x]));
				eeprom_write_block(&R.presetPos[x],
					&E.presetPos[x],
					sizeof (R.presetPos[x]));
			}

			// NEW CODE TO BE WRITTEN XXXXXXXXX
			// sort_and_store_frequency();

			lcdClear();
			lcdGotoXY(0,3);
	   		lcdPrintData("New value stored",16);
			Menu_disp_timer = 50;	// Show on LCD for 5 seconds
			Menu_Mode &=  ~CONFIG;	// We're done
			LCD_upd = FALSE;		// Make ready for next time
		}
		else
		{
			if (current_selection==NUM_PRESETS) // 1st menu pos above memories
												// is selected - Factory Reset
			{
				menu_level = FACTORY_MENU;
				LCD_upd = FALSE;		// force LCD reprint
			}
			else 						// Exit
			{
				lcdClear();
				lcdGotoXY(0,3);
		   		lcdPrintData("Return from Menu",16);
				Menu_disp_timer = 50;	// Show on LCD for 5 seconds
				Menu_Mode &=  ~CONFIG;	// We're done
				LCD_upd = FALSE;		// Make ready for next time
			}
		}
	}
}


//
//--------------------------------------------------------------------
// Scan the Configuraton Menu Status and delegate tasks accordingly
//--------------------------------------------------------------------
//
void PushButtonMenu(void)
{
	// Select which menu level to manage
	if (menu_level == 0) menu_level0();

	//else if (menu_level == PRESET_MENU) preset_menu();

	//else if (menu_level == ENCODER_MENU) encoder_menu();

	//else if (menu_level == DEBUG_MENU) debug_menu();

	else if (menu_level == FACTORY_MENU) factory_menu();
}
